param(
  [string]$VstoUrl = "https://dobid.ljazg.com/addins/setup.exe",
  [bool]$LaunchVsto = $true,
  [bool]$CleanClickOnce = $true
)

$ErrorActionPreference = "Stop"

# 日志文件：脚本同目录
$ScriptDir = Split-Path -Parent $PSCommandPath
$LogPath = Join-Path $ScriptDir ("install_log_{0}.txt" -f (Get-Date -Format "yyyyMMdd_HHmmss"))

function Log([string]$msg) {
  $line = "[{0}] {1}" -f (Get-Date -Format "yyyy-MM-dd HH:mm:ss"), $msg
  Write-Host $line
  Add-Content -Path $LogPath -Value $line -Encoding UTF8
}

function Ensure-Admin {
  $id = [Security.Principal.WindowsIdentity]::GetCurrent()
  $p  = New-Object Security.Principal.WindowsPrincipal($id)
  if (-not $p.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {

    Log "Not elevated. Relaunching as Administrator..."
    $args = @(
      "-NoProfile",
      "-ExecutionPolicy", "Bypass",
      "-File", "`"$PSCommandPath`"",
      "-VstoUrl", "`"$VstoUrl`"",
      "-LaunchVsto", ($LaunchVsto.ToString()),
      "-CleanClickOnce", ($CleanClickOnce.ToString())
    )
    Start-Process -FilePath "powershell.exe" -ArgumentList $args -Verb RunAs | Out-Null
    exit 0
  }
  Log "Running as Administrator."
}

function Close-Office {
  $names = @("WINWORD","EXCEL","POWERPNT","OUTLOOK","ONENOTE","MSACCESS","VISIO")
  foreach ($n in $names) {
    Get-Process -Name $n -ErrorAction SilentlyContinue | Stop-Process -Force -ErrorAction SilentlyContinue
  }
  Log "Office processes closed (if any)."
}

function NormalizeThumb([string]$t) {
  if ($t -eq $null) { $t = "" }
  return ($t -replace " ", "").ToUpperInvariant()
}

function Load-Cert([string]$path) {
  if (-not (Test-Path $path)) { throw "Missing cert file: $path" }
  try {
    return New-Object System.Security.Cryptography.X509Certificates.X509Certificate2($path)
  } catch {
    throw "Failed to load certificate file: $path. $_"
  }
}

function Exists-InStore([string]$storeName, [string]$storeLocation, [string]$thumb) {
  $thumb = NormalizeThumb $thumb
  $store = New-Object System.Security.Cryptography.X509Certificates.X509Store($storeName, $storeLocation)
  $store.Open([System.Security.Cryptography.X509Certificates.OpenFlags]::ReadOnly)
  try {
    foreach ($c in $store.Certificates) {
      if ((NormalizeThumb $c.Thumbprint) -eq $thumb) { return $true }
    }
    return $false
  } finally {
    $store.Close()
  }
}

function Add-WithDotNet([System.Security.Cryptography.X509Certificates.X509Certificate2]$cert, [string]$storeName, [string]$storeLocation) {
  $store = New-Object System.Security.Cryptography.X509Certificates.X509Store($storeName, $storeLocation)
  $store.Open([System.Security.Cryptography.X509Certificates.OpenFlags]::ReadWrite)
  try {
    $store.Add($cert)
  } finally {
    $store.Close()
  }
}

function Add-WithCertutil([string]$certPath, [string]$storeName) {
  # certutil 写 LocalMachine 用 addstore 即可（管理员）
  $cmd = "certutil"
  $args = @("-addstore", "-f", $storeName, "`"$certPath`"")
  Log ("Running: {0} {1}" -f $cmd, ($args -join " "))
  $p = Start-Process -FilePath $cmd -ArgumentList $args -PassThru -Wait -NoNewWindow
  Log ("certutil exit code: {0}" -f $p.ExitCode)
}

function Install-Cert([string]$certPath, [string]$storeName, [string]$storeLocation) {
  $cert = Load-Cert $certPath
  $thumb = NormalizeThumb $cert.Thumbprint

  Log ("---- Installing to {0}\{1} ----" -f $storeLocation, $storeName)
  Log ("File: {0}" -f $certPath)
  Log ("Subject: {0}" -f $cert.Subject)
  Log ("Issuer : {0}" -f $cert.Issuer)
  Log ("Thumb  : {0}" -f $thumb)
  Log ("Valid  : {0} -> {1}" -f $cert.NotBefore, $cert.NotAfter)

  if (Exists-InStore $storeName $storeLocation $thumb) {
    Log "Already exists in store. Skip."
    return
  }

  # 先用 .NET 方式写入
  try {
    Log "Try import via .NET X509Store.Add..."
    Add-WithDotNet $cert $storeName $storeLocation
    Log "DotNet import done."
  } catch {
    Log ("DotNet import FAILED: {0}" -f $_.Exception.Message)

    # 再用 certutil 兜底（很多机器策略不同，成功率更高）
    try {
      Log "Try import via certutil..."
      Add-WithCertutil $certPath $storeName
    } catch {
      Log ("certutil import FAILED: {0}" -f $_.Exception.Message)
    }
  }

  # 导入后再校验一次
  if (Exists-InStore $storeName $storeLocation $thumb) {
    Log "VERIFY OK: certificate is present in target store."
  } else {
    Log "VERIFY FAILED: certificate NOT found in target store."
    throw "Certificate import verification failed for $certPath -> $storeLocation\$storeName"
  }
}

function Clean-ClickOnceCache {
  Log "Cleaning ClickOnce cache..."
  & rundll32.exe dfshim.dll,CleanOnlineAppCache | Out-Null
  Log "ClickOnce cache cleaned."
}

function Dump-Env {
  Log ("PowerShell: {0}" -f $PSVersionTable.PSVersion)
  Log ("OS        : {0}" -f (Get-CimInstance Win32_OperatingSystem | Select-Object -ExpandProperty Caption))
  Log ("Build     : {0}" -f (Get-CimInstance Win32_OperatingSystem | Select-Object -ExpandProperty Version))
  Log ("User      : {0}" -f $env:USERNAME)
}

try {
  Dump-Env
  Ensure-Admin
  Log ("Log file  : {0}" -f $LogPath)

  $rootCer = Join-Path $ScriptDir "CTN2.cer"   # Root: Certum Trusted Network CA 2
  $intCer  = Join-Path $ScriptDir "CCS.cer"    # CA  : Certum Code Signing 2021 CA (intermediate)
  $pubCer  = Join-Path $ScriptDir "YBX.cer"    # TP  : Your publisher cert (Beijing ...)

  Close-Office

  # 证书安装顺序：根 -> 中间 -> 发布者
  Install-Cert $rootCer "Root" "LocalMachine"
  Install-Cert $intCer  "CA" "LocalMachine"
  Install-Cert $pubCer  "TrustedPublisher" "LocalMachine"

  if ($CleanClickOnce) {
    Clean-ClickOnceCache
  }

  Close-Office

  if ($LaunchVsto) {
    Log ("Launching VSTO: {0}" -f $VstoUrl)
    Start-Process -FilePath $VstoUrl
  }

  Log "DONE."
}
catch {
  Log ("FAILED: {0}" -f $_.Exception.Message)
  Log $_.Exception.ToString()
  exit 1
}
